//===============================================================================
// Microsoft patterns & practices Enterprise Library
// Configuration Application Block
//===============================================================================
// Copyright  Microsoft Corporation.  All rights reserved.
// THIS CODE AND INFORMATION IS PROVIDED "AS IS" WITHOUT WARRANTY
// OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT
// LIMITED TO THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
// FITNESS FOR A PARTICULAR PURPOSE.
//===============================================================================

using System;
using System.Reflection;
using Microsoft.Practices.EnterpriseLibrary.Configuration.Factory;
using Microsoft.Practices.EnterpriseLibrary.Configuration.Properties;

namespace Microsoft.Practices.EnterpriseLibrary.Configuration.Injection
{
    /// <summary>
    /// Indicates that the property value is a name reference to another object configuration that can be created through the factory type by name.
    /// </summary>
    /// <remarks>
    /// The factory type for this attribute must be a derived type of <see cref="INamedObjectFactory"/>.
    /// </remarks>
    public class ConverterAttribute : ConverterBaseAttribute
    {
        private readonly Type factoryType;

        /// <summary>
        /// Initialize a new instance of the <see cref="ReferencePropertyParameterAttribute"/> class with the order, parameter type, property name, the factory type and if the value is required.
        /// </summary>
        /// <param name="order">The ordinal position of the constructor parameter.</param>
        /// <param name="parameterType">The parameter type.</param>
        /// <param name="propertyName">The name of the property to get the parameter value.</param>
        /// <param name="factoryType">The <see cref="Type"/> of factory to use to get the value.</param>
        public ConverterAttribute(Type factoryType)
        {
            if (factoryType == null)
            {
                throw new ArgumentNullException("factoryType");
            }

            this.factoryType = factoryType;
        }

        /// <summary>
        /// Gets the factory type to use to create the value.
        /// </summary>
        /// <value>
        /// The factory type to use to create the value.
        /// </value>
        public Type FactoryType
        {
            get { return factoryType; }
        }

        /// <summary>
        /// Gets the <see cref="ReferenceFactory"/> to use to transform the property value to the value for the parameter.
        /// </summary>
        /// <returns>The <see cref="ReferenceFactory"/> to use to transform the property value to the value for the parameter.</returns>
        protected internal override IObjectFactory<TObjectToCreate, TObjectConfiguration> GetFactory<TObjectToCreate, TObjectConfiguration>(TObjectConfiguration objectConfiguration)
        {
            return (IObjectFactory<TObjectToCreate, TObjectConfiguration>)Activator.CreateInstance(this.FactoryType);
        }
    }
}
